<?php
declare(strict_types=1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../middleware/AuthMiddleware.php';

abstract class BaseController
{
    // ── Response helpers ──────────────────────────────────────────────────────

    protected function json(mixed $data, int $status = 200): void
    {
        http_response_code($status);
        echo json_encode($data, JSON_UNESCAPED_UNICODE);
    }

    protected function notFound(string $msg = 'Not found.'): void
    {
        $this->json(['message' => $msg], 404);
    }

    protected function badRequest(string $msg = 'Bad request.'): void
    {
        $this->json(['message' => $msg], 400);
    }

    protected function validationError(array $errors): void
    {
        http_response_code(422);
        echo json_encode([
            'message' => 'Validation failed',
            'errors'  => $errors,
        ], JSON_UNESCAPED_UNICODE);
    }

    protected function serverError(string $msg = 'Server error.'): void
    {
        $this->json(['message' => $msg], 500);
    }

    // ── Request helpers ───────────────────────────────────────────────────────

    /**
     * Parse JSON body or fall back to POST fields.
     */
    protected function body(): array
    {
        $contentType = $_SERVER['CONTENT_TYPE'] ?? '';

        if (str_contains($contentType, 'application/json')) {
            $raw = file_get_contents('php://input');
            return json_decode($raw, true) ?? [];
        }

        return $_POST;
    }

    protected function query(string $key, mixed $default = null): mixed
    {
        return $_GET[$key] ?? $default;
    }

    // ── Auth guard ────────────────────────────────────────────────────────────

    protected function requireAuth(): array
    {
        return AuthMiddleware::handle();
    }

    // ── Validation ────────────────────────────────────────────────────────────

    protected function validate(array $data, array $rules): array
    {
        $errors = [];

        foreach ($rules as $field => $rule) {
            $value = $data[$field] ?? null;
            $parts = explode('|', $rule);

            foreach ($parts as $part) {
                [$ruleName, $ruleArg] = array_pad(explode(':', $part, 2), 2, null);

                switch ($ruleName) {
                    case 'required':
                        if ($value === null || trim((string) $value) === '') {
                            $errors[] = ['field' => $field, 'message' => ucfirst(str_replace('_', ' ', $field)) . ' is required.'];
                        }
                        break;

                    case 'email':
                        if ($value && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
                            $errors[] = ['field' => $field, 'message' => 'Invalid email format.'];
                        }
                        break;

                    case 'int':
                        if ($value !== null && !filter_var($value, FILTER_VALIDATE_INT)) {
                            $errors[] = ['field' => $field, 'message' => ucfirst($field) . ' must be an integer.'];
                        }
                        break;

                    case 'min':
                        if ($value !== null && (int) $value < (int) $ruleArg) {
                            $errors[] = ['field' => $field, 'message' => ucfirst($field) . " must be at least {$ruleArg}."];
                        }
                        break;

                    case 'in':
                        $allowed = explode(',', $ruleArg);
                        if ($value !== null && !in_array($value, $allowed, true)) {
                            $errors[] = ['field' => $field, 'message' => ucfirst($field) . ' has an invalid value.'];
                        }
                        break;

                    case 'phone':
                        if ($value && !preg_match('/^[0-9+\-() ]{8,15}$/', $value)) {
                            $errors[] = ['field' => $field, 'message' => 'Invalid phone number format.'];
                        }
                        break;
                }
            }
        }

        return $errors;
    }
}
