<?php
declare(strict_types=1);

require_once __DIR__ . '/BaseController.php';
require_once __DIR__ . '/../models/RouteModel.php';

class RouteController extends BaseController
{
    public function index(array $params): void
    {
        $this->json(RouteModel::getAll());
    }

    public function show(array $params): void
    {
        $route = RouteModel::getById((int) $params['id']);
        if (!$route) {
            $this->notFound('Route not found.');
            return;
        }
        $this->json($route);
    }

    public function store(array $params): void
    {
        $this->requireAuth();
        $body   = $this->body();
        $errors = $this->validate($body, ['nama_trayek' => 'required']);

        if ($errors) {
            $this->validationError($errors);
            return;
        }

        $id    = RouteModel::create(trim($body['nama_trayek']));
        $route = RouteModel::getById($id);
        $this->json($route, 201);
    }

    public function update(array $params): void
    {
        $this->requireAuth();
        $body   = $this->body();
        $errors = $this->validate($body, ['nama_trayek' => 'required']);

        if ($errors) {
            $this->validationError($errors);
            return;
        }

        $affected = RouteModel::update((int) $params['id'], trim($body['nama_trayek']));
        if (!$affected) {
            $this->notFound('Route not found.');
            return;
        }

        $this->json(RouteModel::getById((int) $params['id']));
    }

    public function destroy(array $params): void
    {
        $this->requireAuth();

        try {
            $affected = RouteModel::delete((int) $params['id']);
            if (!$affected) {
                $this->notFound('Route not found.');
                return;
            }
            $this->json(['message' => 'Route deleted successfully.']);
        } catch (RuntimeException $e) {
            $this->json(['message' => $e->getMessage()], 409);
        }
    }
}
