<?php
declare(strict_types=1);

require_once __DIR__ . '/BaseController.php';
require_once __DIR__ . '/../models/SubmissionModel.php';
require_once __DIR__ . '/../services/RegistrationService.php';
require_once __DIR__ . '/../services/ZipService.php';

class SubmissionController extends BaseController
{
    private const REQUIRED_FILES = [
        'supervision_card',
        'stnk',
        'inspection_book',
        'business_license',
        'insurance',
        'ktp',
        'retribution_proof',
    ];

    private const MAX_FILE_SIZE = 1048576; // 1 MB in bytes

    // ── POST /api/submissions ─────────────────────────────────────────────────
    public function store(array $params): void
    {
        // Validate text fields
        $errors = $this->validate($_POST, [
            'route_id'     => 'required|int|min:1',
            'owner_name'   => 'required',
            'phone'        => 'required|phone',
            'plate_number' => 'required',
            'address'      => 'required',
        ]);

        // Validate files
        foreach (self::REQUIRED_FILES as $field) {
            if (empty($_FILES[$field]['name'])) {
                $errors[] = ['field' => $field, 'message' => ucfirst(str_replace('_', ' ', $field)) . ' document is required.'];
                continue;
            }

            if ($_FILES[$field]['error'] !== UPLOAD_ERR_OK) {
                $err = $_FILES[$field]['error'];
                if ($err === UPLOAD_ERR_INI_SIZE || $err === UPLOAD_ERR_FORM_SIZE) {
                    $errors[] = ['field' => $field, 'message' => "File {$field} exceeds 1MB limit."];
                } else {
                    $errors[] = ['field' => $field, 'message' => "Upload error for {$field}."];
                }
                continue;
            }

            if ($_FILES[$field]['size'] > self::MAX_FILE_SIZE) {
                $errors[] = ['field' => $field, 'message' => "File {$field} exceeds 1MB limit."];
                continue;
            }

            // Validate MIME type (check both MIME and file signature)
            $finfo    = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $_FILES[$field]['tmp_name']);
            finfo_close($finfo);

            if ($mimeType !== 'application/pdf') {
                $errors[] = ['field' => $field, 'message' => "File {$field} must be a PDF."];
            }
        }

        if ($errors) {
            $this->validationError($errors);
            return;
        }

        // Generate registration number
        $regNumber = RegistrationService::generate();
        $destDir   = __DIR__ . "/../uploads/pengajuan/{$regNumber}";

        if (!mkdir($destDir, 0755, true) && !is_dir($destDir)) {
            $this->serverError('Could not create upload directory.');
            return;
        }

        // Move uploaded files
        $fileNames = [];
        foreach (self::REQUIRED_FILES as $field) {
            $newName = "{$field}_" . time() . ".pdf";
            $dest    = "{$destDir}/{$newName}";

            if (!move_uploaded_file($_FILES[$field]['tmp_name'], $dest)) {
                $this->serverError("Could not save file: {$field}.");
                return;
            }

            $fileNames[$field] = $newName;
        }

        // Save to DB
        $submissionId = SubmissionModel::create([
            'registration_number' => $regNumber,
            'route_id'            => $_POST['route_id'],
            'owner_name'          => trim($_POST['owner_name']),
            'phone'               => trim($_POST['phone']),
            'plate_number'        => strtoupper(trim($_POST['plate_number'])),
            'address'             => trim($_POST['address']),
            ...$fileNames,
        ]);

        $this->json([
            'message'             => 'Submission created successfully.',
            'registration_number' => $regNumber,
            'id'                  => $submissionId,
        ], 201);
    }

    // ── GET /api/submissions ──────────────────────────────────────────────────
    public function index(array $params): void
    {
        $this->requireAuth();

        $submissions = SubmissionModel::getAll([
            'status'   => $this->query('status'),
            'route_id' => $this->query('route_id'),
        ]);

        $this->json($submissions);
    }

    // ── GET /api/submissions/dashboard ───────────────────────────────────────
    public function dashboardStats(array $params): void
    {
        $this->requireAuth();

        $stats = SubmissionModel::getDashboardStats();
        $chart = SubmissionModel::getDailyChart(30);

        // Cast numerics
        $stats['total']    = (int) $stats['total'];
        $stats['pending']  = (int) $stats['pending'];
        $stats['approved'] = (int) $stats['approved'];
        $stats['rejected'] = (int) $stats['rejected'];

        foreach ($chart as &$row) {
            $row['count'] = (int) $row['count'];
        }

        $this->json(compact('stats', 'chart'));
    }

    // ── GET /api/submissions/:id ──────────────────────────────────────────────
    public function show(array $params): void
    {
        $this->requireAuth();

        $submission = SubmissionModel::getById((int) $params['id']);
        if (!$submission) {
            $this->notFound('Submission not found.');
            return;
        }

        $this->json($submission);
    }

    // ── PATCH /api/submissions/:id/status ────────────────────────────────────
    public function updateStatus(array $params): void
    {
        $this->requireAuth();

        $body   = $this->body();
        $errors = $this->validate($body, [
            'status' => 'required|in:pending,approved,rejected',
        ]);

        if ($errors) {
            $this->validationError($errors);
            return;
        }

        $affected = SubmissionModel::updateStatus(
            (int) $params['id'],
            $body['status'],
            $body['admin_notes'] ?? null
        );

        if (!$affected) {
            $this->notFound('Submission not found.');
            return;
        }

        $this->json(SubmissionModel::getById((int) $params['id']));
    }

    // ── GET /api/submissions/:id/files/:field/preview ─────────────────────────
    public function previewFile(array $params): void
    {
        $this->requireAuth();
        $this->serveFile($params, inline: true);
    }

    // ── GET /api/submissions/:id/files/:field/download ────────────────────────
    public function downloadFile(array $params): void
    {
        $this->requireAuth();
        $this->serveFile($params, inline: false);
    }

    // ── GET /api/submissions/:id/download-zip ────────────────────────────────
    public function downloadZip(array $params): void
    {
        $this->requireAuth();

        $submission = SubmissionModel::getById((int) $params['id']);
        if (!$submission) {
            $this->notFound('Submission not found.');
            return;
        }

        // ZipService sends its own headers and output
        ZipService::streamSubmission($submission);
    }

    // ── GET /api/submissions/track/:reg_number (public) ──────────────────────
    public function trackByRegNumber(array $params): void
    {
        $submission = SubmissionModel::getByRegistrationNumber(
            strtoupper(trim($params['reg_number']))
        );

        if (!$submission) {
            $this->notFound('Submission not found.');
            return;
        }

        // Return only public-safe fields
        $this->json([
            'registration_number' => $submission['registration_number'],
            'owner_name'          => $submission['owner_name'],
            'phone'               => $submission['phone'],
            'plate_number'        => $submission['plate_number'],
            'nama_trayek'         => $submission['nama_trayek'],
            'status'              => $submission['status'],
            'admin_notes'         => $submission['admin_notes'],
            'created_at'          => $submission['created_at'],
            'updated_at'          => $submission['updated_at'],
        ]);
    }

    // ── Private: serve a single PDF file ─────────────────────────────────────
    private function serveFile(array $params, bool $inline): void
    {
        $submission = SubmissionModel::getById((int) $params['id']);
        if (!$submission) {
            $this->notFound('Submission not found.');
            return;
        }

        $field = $params['field'];
        if (!in_array($field, self::REQUIRED_FILES, true)) {
            $this->badRequest('Invalid file field.');
            return;
        }

        $filename = $submission[$field] ?? null;
        if (!$filename) {
            $this->notFound('File not found.');
            return;
        }

        $filePath = __DIR__ . "/../uploads/pengajuan/{$submission['registration_number']}/{$filename}";

        if (!file_exists($filePath)) {
            $this->notFound('File not found on disk.');
            return;
        }

        $disposition = $inline ? 'inline' : 'attachment';

        // Clear any previously set JSON header
        header('Content-Type: application/pdf');
        header("Content-Disposition: {$disposition}; filename=\"{$field}.pdf\"");
        header('Content-Length: ' . filesize($filePath));
        header('Cache-Control: no-cache');

        readfile($filePath);
    }
}
